/**
* \file: AditAudioSource.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Audio Source Endpoint interfaces
*
* \component: Baidu CarLife
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef BDCL_ADITAUDIOSOURCE_H
#define BDCL_ADITAUDIOSOURCE_H

#include <string>
#include "bdclErrorCodes.h"

namespace adit { namespace bdcl {

class CoreCallbackDealer;

/** @interface  IAditAudioSourceCallbacks
 *  @brief      Callbacks to notify the MC application. Most of the callbacks are initiated by Baidu stack and match
 *              their signature. This interface must be implemented by the MC application using ADIT Audio Source
 *              implementation.
 */
class IAditAudioSourceCallbacks
{
public:
    virtual ~IAditAudioSourceCallbacks() { }

    /** @brief Callback initiated by Baidu core to start voice wakeup.
     */
    virtual void onMicRecordWakeupStart() { };

    /** @brief Callback initiated by Baidu core to end voice recording.
     */
    virtual void onMicRecordEnd() { };

    /** @brief Callback initiated by Baidu core to enter voice recognition phase.
     */
    virtual void onMicRecordRecognitionStart() { };

    /** @brief Callback initiated by ADIT to inform MC application about an error.
     */
    virtual void onError(bdclErrorCodes inErrorCode) { (void) inErrorCode; };
};

/** @class      AditAudioSource
 *  @brief      Abstract audio source interface to be used by the MC. MC has to implement a class of a concrete
 *              implementation of AditAudioSource, but use AditAudioSource interfaces for easier portability.
 */
class AditAudioSource
{
public:
    // shared ptr, no document
    // after the teardown is called, it has to be ensured that these callbacks won't be triggered anymore.
    // So MC is free to destroy them, but MC has to guarantee that the IAditAudioSinkCallbacks object is alive before
    // they call the teardown.

    /** @brief AditAudioSource constructor
     *  @param inCallbacks: object implementing the callback interface. MC has to ensure that IAditAudioSourceCallbacks
     *         object is alive until they call teardown.
     */
    AditAudioSource(IAditAudioSourceCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
    {
        (void)inCallbacks;
        (void)inCallbackDealer;
    }

    virtual ~AditAudioSource() { };

    // todo write the config params as comment: mediaType,
    //      std::string deviceName, int bufferSizeMin, int bufferSizeMax, int threadPriority, int initTimeOut
    //      unsigned int inSampleRate, unsigned int inChannelConfig, unsigned int inSampleFormat
    /** @brief Configuration interface for each parameter
     *  @param inKey: name of the configuration parameter
     *  @param inValue: the actual value for configuration
     */
    virtual void setConfigItem(std::string inKey, std::string inValue) = 0;

    /** @brief Method to initialize audio sink endpoint
     *         TODO: decide whether initialize() / teardown() should be accessible by MC. The alternative would be MC
     *         only accessing the abdcl interface for initializing or ending whole session, which will initialize / end
     *         audio session
     *  @return true on success
     *          false on failure
     */
    virtual bool initialize() = 0;
    /** @brief Method to shutdown audio sink endpoint. After teardown call, MC is free to destroy
     *         IAditAudioSinkCallbacks object
     */
    virtual void teardown() = 0;

    /** @brief Method to start audio playback. MC has to ensure that the configuration is done before calling starting
     *         audio playback.
     *  @return true on success
     *          false on failure
     */
    virtual bool captureStart() = 0;
    /** @brief Method to stop audio playback.
     *  @param inFlushBuffer: flag to clean prebuffered audio when stopping audio playback
     *  @return true on success
     *          false on failure
     */
    virtual bool captureStop() = 0;

};

} } /* namespace adit { namespace bdcl { */

#endif /* BDCL_ADITAUDIOSOURCE_H */
